<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package StudioPress\Genesis
 * @author  StudioPress
 * @license GPL-2.0-or-later
 * @link    https://my.studiopress.com/themes/genesis/
 */

/**
 * Generates the SEO Document Title for Genesis.
 *
 * @since 2.6.0
 *
 * @package Genesis\SEO
 */
class Genesis_SEO_Document_Title_Parts {

	/**
	 * Key used for title part.
	 *
	 * @since 2.6.0
	 */
	const TITLE = 'title';

	/**
	 * Key used for tagline part.
	 *
	 * @since 2.6.0
	 */
	const TAGLINE = 'tagline';

	/**
	 * Key used for site name part.
	 *
	 * @since 2.6.0
	 */
	const SITE = 'site';

	/**
	 * Document title parts.
	 *
	 * @since 2.6.0
	 *
	 * @var array
	 */
	protected $parts;

	/**
	 * Constructor.
	 *
	 * @since 2.6.0
	 */
	public function __construct() {

		$this->parts = $this->get_default_parts();

	}

	/**
	 * Get title parts array.
	 *
	 * Uses context to return document title parts.
	 *
	 * @since 2.6.0
	 *
	 * @param array $parts Optional. A default $parts array that can be used as a fallback.
	 * @return array Return a contextual array of document title parts.
	 */
	public function get_parts( array $parts = [] ) {

		$this->parts = wp_parse_args( $parts, $this->parts );

		if ( genesis_is_root_page() ) {
			$this->parts = $this->get_root_page_title_parts();
		}

		if ( is_home() ) {
			$this->parts = $this->get_home_page_title_parts();
		}

		if ( is_singular() ) {
			$this->parts = $this->get_singular_title_parts();
		}

		if ( is_category() || is_tag() || is_tax() ) {
			$this->parts = $this->get_tax_archive_title_parts();
		}

		if ( is_author() ) {
			$this->parts = $this->get_author_archive_title_parts();
		}

		if ( is_post_type_archive() ) {
			$this->parts = $this->get_post_type_archive_title_parts();
		}

		if ( is_feed() ) {
			$this->parts = $this->get_default_parts();
		}

		if ( ! genesis_get_seo_option( 'append_site_title' ) ) {
			unset( $this->parts[ self::SITE ] );
		}

		return $this->parts;

	}

	/**
	 * Get title parts for root page.
	 *
	 * @since 2.6.0
	 *
	 * @return array Document title parts.
	 */
	public function get_root_page_title_parts() {

		$home_doctitle      = genesis_get_seo_option( 'home_doctitle' );
		$append_description = genesis_get_seo_option( 'append_description_home' );

		$this->parts[ self::TITLE ] = $home_doctitle ?: $this->parts[ self::TITLE ];

		if ( ! $append_description ) {
			unset( $this->parts[ self::TAGLINE ] );
		}

		return $this->parts;

	}

	/**
	 * Get title parts for home page.
	 *
	 * @since 2.6.0
	 *
	 * @return array Document title parts.
	 */
	public function get_home_page_title_parts() {

		$page_for_posts = get_option( 'page_for_posts' );

		if ( $page_for_posts && get_queried_object_id() ) {
			$this->parts = $this->get_singular_title_parts( $page_for_posts );
		}

		return $this->parts;

	}

	/**
	 * Get title parts for singular entries.
	 *
	 * @since 2.6.0
	 *
	 * @param int $post_id Optional. Post ID. Default is null.
	 * @return array Document title parts.
	 */
	public function get_singular_title_parts( $post_id = null ) {

		$supported_title_keys = [
			'_genesis_title',
			'_aioseo_title',
			'_headspace_title',
			'thesis_title',
			'title_tag',
			'title',
		];

		foreach ( $supported_title_keys as $key ) {
			if ( genesis_get_custom_field( $key, $post_id ) ) {
				$this->parts[ self::TITLE ] = genesis_get_custom_field( $key, $post_id );
				break;
			}
		}

		return $this->parts;

	}

	/**
	 * Get title parts for taxonomy archives.
	 *
	 * @since 2.6.0
	 *
	 * @return array Document title parts.
	 */
	public function get_tax_archive_title_parts() {

		$term       = get_queried_object();
		$term_title = get_term_meta( $term->term_id, 'doctitle', true );

		if ( ! empty( $term_title ) ) {
			$this->parts[ self::TITLE ] = $term_title;
		}

		return $this->parts;

	}

	/**
	 * Get title parts for author archives.
	 *
	 * @since 2.6.0
	 *
	 * @return array Document title parts.
	 */
	public function get_author_archive_title_parts() {

		$user_title = get_the_author_meta( 'doctitle', (int) get_query_var( 'author' ) );

		if ( ! empty( $user_title ) ) {
			$this->parts[ self::TITLE ] = $user_title;
		}

		return $this->parts;

	}

	/**
	 * Get title parts for post type archives.
	 *
	 * @since 2.6.0
	 *
	 * @return array Document title parts.
	 */
	public function get_post_type_archive_title_parts() {

		if ( genesis_has_post_type_archive_support() ) {

			$cpt_title = genesis_get_cpt_option( 'doctitle' );

			if ( $cpt_title ) {
				$this->parts[ self::TITLE ] = $cpt_title;
			}
		}

		return $this->parts;

	}

	/**
	 * Set default document title parts array.
	 *
	 * @since 2.6.0
	 *
	 * @return array Document title parts.
	 */
	public function get_default_parts() {
		return [
			'title' => get_bloginfo( 'name', 'display' ),
		];
	}

}
